## ----global_options, include=FALSE--------------------------------------------
knitr::opts_chunk$set(out.extra = '', fig.pos = 'H', collapse = TRUE, comment = NA)

## ----launchoverview1, eval = FALSE--------------------------------------------
#  library(StrathE2E2)     # Load the package
#  help(StrathE2E2)        # Launch the overview document in a browser window

## ----launchoverview2, eval = TRUE---------------------------------------------
citation("StrathE2E2")

## ----quickstart, fig.width=7, fig.height=6, fig.align='center'----------------
library(StrathE2E2)				# Load the package
model <- e2e_read("North_Sea", "1970-1999")	# Read the 1970-1999 version of the North Sea model
results <- e2e_run(model,nyears=3)		# Run the model for 3 years
e2e_plot_ts(model,results,"ECO")	        # Plot time series of the masses of the state variables

## ----loadpackage--------------------------------------------------------------
library(StrathE2E2)

## ----listmodels---------------------------------------------------------------
library("StrathE2E2")
e2e_ls()

## ----copymodels, eval = FALSE-------------------------------------------------
#  library("StrathE2E2")
#  e2e_copy(" Modelname " , " Modelvariant ", dest.path= " mymodels ", user.path= " source " )
#  #
#  # The following call would copy one of the example models in he package to a user location:
#  e2e_copy("North_Sea" , "1970-1999", dest.path= " mymodels " )

## ----getparmdoc---------------------------------------------------------------
parm_list <- e2e_get_parmdoc(0)       # Get documentation on the fitted ecology parameters
parm_list <- e2e_get_parmdoc(2)       # Get documentation on the fishing fleet model parameters
parm_list <- e2e_get_parmdoc()        # Get documentation all parameters
head(parm_list)

## ----loadmodel----------------------------------------------------------------
library("StrathE2E2")
model <- e2e_read("North_Sea", "1970-1999")

## ----viewmodelobject----------------------------------------------------------
str(model,max.level=2)                    # View the top 2 levels of input list object

## ----runmodel-----------------------------------------------------------------
results <- e2e_run(model, nyears=3)         # Run the model for 3 years
str(results,max.level=1)                    # View the high-level contents of the output list object

## ----extractendcon, eval=FALSE------------------------------------------------
#  model<-e2e_read("North_Sea", "2003-2013")                             # Read the 2003-2013 internal model
#  results <- e2e_run(model, nyears=3)                                   # Run the model for 3 years
#  new_start_data <- e2e_extract_start(model,results,csv.output=FALSE)   # Extract the model endstate as a dataframe only
#  # This will produce the new data only as a dataframe
#  head(new_start_data)
#  #
#  #
#  # Copy the 2003-2013 version of the North Sea model supplied with the package into a user workspace (Windows OS):
#  e2e_copy("North_Sea", "2003-2013",
#              dest.path="C:/Users/username/Documents/Models")
#  # Load the 2003-2013 version of the North Sea model from the user workspace:
#  model<-e2e_read(model.name="North_Sea",
#                    model.variant="2003-2013",
#                    model.ident="TEST",
#                    user.path="C:/Users/username/Documents/Models")
#  results <- e2e_run(model, nyears=3)                    # Run the model for 3 years
#  new_start_data <- e2e_extract_start(model,results)     # Extract the model endstate as new initial conditions file
#  # This will write to the user model /Parameters folder)

## ----extractharvestratios, eval=TRUE------------------------------------------
# Load the 2003-2013 version of the North Sea model from the user workspace:
model<-e2e_read(model.name="North_Sea",
                  model.variant="2003-2013")
results <- e2e_run(model, nyears=3)                                   # Run the model for 3 years
harvest_ratio_data <- e2e_extract_hr(model,results,csv.output=FALSE)  # Extract the harvest ratio data
harvest_ratio_data

## ----plot_ts_01, fig.width=7, fig.height=6, fig.align='center'----------------
model<-e2e_read(model.name="North_Sea",
                  model.variant="1970-1999")
results <- e2e_run(model, nyears=3)
e2e_plot_ts(model, results,selection="ECO")

## ----plot_ts_02, fig.width=7, fig.height=6, fig.align='center'----------------
e2e_plot_ts(model, results,selection="CATCH")

## ----plot_ts_02_tofile, eval = FALSE------------------------------------------
#  pdf("plot.pdf",width=7,height=6)                    # or jpeg("plot.jpg"), png("plot.PNG")
#  e2e_plot_ts(model, results,selection="CATCH")
#  dev.off()

## ----viewmodelobject2---------------------------------------------------------
model<-e2e_read("North_Sea", "2003-2013")
str(model,max.level=2)                    # View the top 2 levels of input list object

## ----viewfleetmodelinput------------------------------------------------------
model<-e2e_read("North_Sea", "2003-2013")
model$data$fleet.model

## ----runmodelscenario1-1, fig.width=7, fig.height=6, fig.align='center'-------
# Example of code to run a baseline case of the North Sea model with 2003-2013 conditions, and then edit the
# model list object to create a scenario run with the temperature in all zones increased by 2 deg-C
#--------------------------------------------------------
# Read the embedded North Sea 2003-2013 model and assign an identifier for the results
baseTemp_model<-e2e_read("North_Sea", "2003-2013",model.ident="baseTemp")
# Run the model for 10 years and save the results to a named list object
baseTemp_results<-e2e_run(baseTemp_model,nyears=10)
# Visualize the output from the run (should show a repeating annual cycle with no trend)
e2e_plot_ts(baseTemp_model,baseTemp_results,"ECO")

## ----runmodelscenario1-2, fig.width=7, fig.height=6, fig.align='center'-------
T_rise<-2    # temperature increase to add to all temperatures in the model drivers
# copy the baseline model list object to a new model list
baseTemp_plusTC_model<-baseTemp_model
# add temperature increase to upper layer offshore temperatures
baseTemp_plusTC_model$data$physics.drivers$so_temp <- baseTemp_model$data$physics.drivers$so_temp+T_rise
# add temperature increase to inshore temperatures
baseTemp_plusTC_model$data$physics.drivers$si_temp <- baseTemp_model$data$physics.drivers$si_temp+T_rise
# add temperature increase to lower layer offshore temperatures
baseTemp_plusTC_model$data$physics.drivers$d_temp  <- baseTemp_model$data$physics.drivers$d_temp+T_rise
# Assign a unique identifier for the .csv outputs
baseTemp_plusTC_model$setup$model.ident <- "baseTemp_plusTC"
# Run the model for 10 years and save the results to a named list object
baseTemp_plusTC_results<-e2e_run(baseTemp_plusTC_model,nyears=10)
# Visualize the output from the run (should show trends in outputs due to change in T)
e2e_plot_ts(baseTemp_plusTC_model,baseTemp_plusTC_results,"ECO")

## ----runmodelscenario2, eval = FALSE------------------------------------------
#  # Example of code to loop through a set of seven levels of demersal fish harvest ratio ranging from 0 to 3-times
#  # the baseline value for the 1970-1999 North Sea model. The .csv outputs for each level are saved to unique
#  # filenames but the results list-object is discarded. The baseline case is the third level (HScale=1). Each level
#  # is run for 40 years.
#  #--------------------------------------------------------
#  model<-e2e_read("North_Sea", "1970-1999")  		# Read the embedded North Sea 1970-1999 model
#  for(i in 1:7) {                              		# Start loop through seven model scenarios
#    HScale <- (i-1)*0.5                          		# HScale becomes 0 to 3 in steps of 0.5
#        # Assign a unique identifier for each set of .csv outputs
#    model$setup$model.ident <- paste("Dem_HR_scale_",HScale,sep="")
#        # Set the demersal fish harvest ratio multipler to HScale
#    model$data$fleet.model$HRscale_vector_multiplier[2] <- HScale
#    results<-e2e_run(model,nyears=40)			 # Run the model, save the .csv results, overwrite any previous list-object
#    cat("Harvest Ratio scaling = ",HScale,"\n")            # Print a screen message to monitor progress
#    e2e_plot_ts(model,results,"ECO")	  	         # Visualize the output from each run
#  }							 # End loop through scenarios

## ----runmodel3----------------------------------------------------------------
model<-e2e_read("North_Sea", "2003-2013")
results <- e2e_run(model, nyears=3)         # Run the model for 3 years
str(results,max.level=1)                    # View the high-level contents of the output list object

## ----resultsobjectdata1-------------------------------------------------------
str(results$build,max.level=1) 

## ----resultsobjectdata2-------------------------------------------------------
head(results$output[,1:6])
tail(results$output[,1:6])

## ----resultsobjectdata3-------------------------------------------------------
# List the names of the first 40 vectors in the results$aggregates list:
names(results$aggregates[1:40])

## ----plotagg, fig.width=6, fig.height=4, fig.align='center'-------------------
# Plot the time series of the sedient nitrate mass in the inshore and offshore zones
# i.e. aggregated across the sediment types in each zone.
# Note that this is the mass not the porewater concentration so uncorrected for
# zonal area or sediment porosity or layer thickness.
plot(seq(1,nrow(results$output)), results$aggregates$x_nitrate_o, type="l",col="black",
     ylim=( c( 0,(1.5*max(results$aggregates$x_nitrate_o)) ) ),
     xlab="Days", ylab="Sediment nitrate mass (mMN)")
lines(seq(1,nrow(results$output)), results$aggregates$x_nitrate_i, type="l",col="red")
legend("topleft",  bg="transparent", c("offshore","inshore"), lty=c(1,1),col=c("black","red"))

## ----resultsobjectdata4-------------------------------------------------------
str(results$fleet.output,max.level=1)

## ----resultsobjectdata5-------------------------------------------------------
str(results$total.annual.catch,max.level=1)

## ----resultsobjectdata6-------------------------------------------------------
str(results$annual.catch.by.gear,max.level=1)

## ----resultsobjectdata7-------------------------------------------------------
str(results$final.year.output,max.level=1)

## ----resultsobjectdata8-------------------------------------------------------
# List the first six entries
str(results$final.year.output[1:6],max.level=1)
#
# View the first two column of the inshore_catchmat array
results$final.year.output$inshore_catchmat[,1:2]

## ----resultsobjectdata9-------------------------------------------------------
results$final.year.output$monthly.averages[,1:2]

## ----resultsobjectdata10------------------------------------------------------
str(results$final.year.output[8:11],max.level=1)
#
head(results$final.year.output$mass_results_inshore)
#
head(results$final.year.output$annual_flux_results_inshore)

## ----resultsobjectdata11------------------------------------------------------
str(results$final.year.output[12:15],max.level=1)

## ----resultsobjectdata12------------------------------------------------------
str(results$final.year.output[16:19],max.level=1)

## ----resultsobjectdata13------------------------------------------------------
str(results$final.year.output[20:22],max.level=1)
#
#Show the first 5 rows and columns of the matrix of all flows:
results$final.year.output$flow_matrix_all_fluxes[1:5,1:5]
#
#Show the first few rows of the dataframe containing outputs from the NetIndices package:
head(results$final.year.output$NetworkIndexResults)

## ----resultsobjectdata14------------------------------------------------------
str(results$final.year.output[23:24],max.level=1)

## ----resultsobjectdata15------------------------------------------------------
str(results$final.year.output[25:27],max.level=1)
#
#Show the overall likelihood value:
results$final.year.output$annual_obj
#
#Show the first few rows of the partial likelihood data:
head(results$final.year.output$partial_chi)
#
#Show the first few rows of the dataframe containing the model outputs corresponding to the annual observational target data:
head(results$final.year.output$opt_results)

## ----processresults1----------------------------------------------------------
out <- results$output

## ----processresults2----------------------------------------------------------
# All the output corresponding to year Y
Y <- 3                    # select year 3
yearY_data <- out[(((Y-1)*360)+1) : ((Y*360)+1),]
#
#List the first few rows and columns of these extracted data
head(yearY_data[,1:6])

## ----processresults3----------------------------------------------------------
# All the output for a given state variable (denoted by "column_name") in year Y:
Y <- 3                    # select year 3
column_name<- "herb_o"    # Select offshore omnivorous zooplankton
col2get <- which(colnames(out)==column_name)
column_data_yearY <- out[(((Y-1)*360)+1) : ((Y*360)+1),col2get]

## ----processresults4----------------------------------------------------------
# Calculate the annual average in year Y of the variable extracted in the previous step:
	mean(column_data_yearY)

## ----processresults5, fig.width=6, fig.height=5, fig.align='center'-----------
column_name<- "phytgrossprod_o"    		   # Select for example offshore gross production by phytoplankton
col2get <- which(colnames(out)==column_name)
J <- nrow(out)					   # number of rows of data in the "out" dataframe
temp <- rep(0,J)				   # creates a vector to hold temporary data
temp[1: (J-1)] <- out[2 : J, col2get]		   # copies data from the "out" into "temp" offset by 1 time increment
temp[J] <- 2*(out[J,col2get]) - out[J-1,col2get]   # fills in the terminal value of the "temp" vector by extrapolation
rate <- temp - out[,col2get]			   # "rate" is a vector of the increments between successive time steps
#
#Plot the original cumulative data and the derived rate
par(mfrow=c(2,1))
par(mar=c(4,4,1,1))
plot(seq(1,length(out[,col2get])),out[,col2get],type="l",xlab="Days",ylab="Cumulative flux")
plot(seq(1,length(rate)),rate,type="l",xlab="Days",ylab="Daily flux")

## ----processresults6----------------------------------------------------------
Y <- 3                                                  # select year 3
column_name<- "phytgrossprod_o"    		        # Select for example offshore gross production by phytoplankton
col2get <- which(colnames(out)==column_name)
out[((Y*360)+1),col2get] - out[(((Y-1)*360)+1),col2get] # integrated flux of the variable denoted by "column_name" over year Y

## ----processresults7----------------------------------------------------------
# Extract the relevant area and volumetric parameters from the $build$model.parameters object
inshore_area 	     <- as.numeric(results$build$model.parameters["shallowprop"])
offshore_upper_thick <- as.numeric(results$build$model.parameters["thik_so"])
offshore_lower_thick <- as.numeric(results$build$model.parameters["thik_d"])
inshore_sed1_area    <- as.numeric(results$build$model.parameters["area_s1"])
inshore_sed1_thick   <- as.numeric(results$build$model.parameters["thik_x_s1"])
inshore_sed1_poros   <- as.numeric(results$build$model.parameters["porosity_s1"])
#
# Time series of suspended detritus and bacteria concentration in the offshore upper layer (mMN.m-3)
detr_bact_conc_so<- out$detritus_so/((1-inshore_area)*offshore_upper_thick)
#
# Time series of the area-density of omnivorous zooplankton in the inshore zone (mMN.m-2)
oz_dens_i<- out$herb_i/inshore_area		
#
# Time series of the depth averaged concentration of omnivorous zooplankton in the offshore zone (mMN.m-3)
oz_conc_o<- out$herb_o /((1-inshore_area)*(offshore_upper_thick+offshore_lower_thick))
#
# Time series of the porewater concentration of ammonia in inshore sediment habitat 1 (mMN.m-3):
xamm_conc_s1<- out$x_ammonia_s1/(inshore_sed1_area*inshore_sed1_thick*inshore_sed1_poros)
#
# Time series of labile detritus & bacterial in inshore sediment habitat 1 as a % of sediment dry weight (%gN.g-1):
xdet_conc_s1<- 100*(out$xR_detritus_s1*(14/1000))/(inshore_sed1_area*inshore_sed1_thick*(1-inshore_sed1_poros)*2650000 )
# (nitrogen atomic weight = 14 g.mole-1; dry sediment density = quartz density = 2.65 x 1e6 g.m-3)	

## ----annealingcode1, eval = FALSE---------------------------------------------
#  # Copy the 2003-2013 version of the North Sea model supplied with the package into a user workspace (Windows OS):
#  	e2e_copy("North_Sea", "2003-2013",
#  	dest.path="C:/Users/username/Documents/Models")
#  #
#  # Load the 2003-2013 version of the North Sea model from the user workspace:
#  	model<-e2e_read(model.name="North_Sea",
#                            model.variant="2003-2013",
#                            model.ident="TEST",
#                            user.path="C:/Users/username/Documents/Models")
#  #
#  # Quick demonstration of the annealing function in operation:
#  e2e_optimize_eco(model, nyears=5, n_iter=10, start_temperature=0.5)
#  #
#  # More realistic configuration would be as follows :
#  e2e_optimize_eco(model, nyears=50, n_iter=1000, start_temperature=1)
#  # (WARNING - this will take about 26 hours to run)
#  #

## ----annealingcode2, eval = FALSE---------------------------------------------
#  # Copy the 2003-2013 version of the North Sea model supplied with the package into a user workspace (Windows OS):
#  	e2e_copy("North_Sea", "2003-2013",
#  	dest.path="C:/Users/username/Documents/Models")
#  #
#  # Load the 2003-2013 version of the North Sea model from the user workspace:
#  	model<-e2e_read(model.name="North_Sea",
#                            model.variant="2003-2013",
#                            model.ident="TEST",
#                            user.path="C:/Users/username/Documents/Models")
#  #
#  # Quick demonstration of the annealing function in operation:
#  e2e_optimize_hr(model, nyears=5, n_iter=10, start_temperature=0.5)
#  #
#  # More realistic configuration would be as follows :
#  e2e_optimize_hr(model, nyears=50, n_iter=1000, start_temperature=1)
#  # (WARNING - this will take about 26 hours to run)
#  #

## ----annealingcode4, eval = FALSE---------------------------------------------
#  # Copy the 1970-1999 version of the North Sea model supplied with the package into a user workspace (Windows OS):
#  e2e_copy("North_Sea", "1970-1999",
#              dest.path="C:/Users/username/Documents/Models")
#  #
#  # Load the 1970-1999 version of the North Sea model from a user workspace:
#  model<-e2e_read(model.name="North_Sea",
#                    model.variant="1970-1999",
#                    model.ident="TEST",
#                    user.path="C:/Users/username/Documents/Models")
#  # Quick Demonstration of optimizing to ecosystem data:
#  test_run <- e2e_optimize_act(model, selection="ECO", n_iter=10, start_temperature=0.4 , cooling=0.975, csv.output=FALSE, nyears=5 )
#  str(test_run,max.level=1)                         # View the structure of the returned list
#  #
#  # More realistic configuration would be (WARNING - this will take about 14 hours to run) :
#  find_gear_activity_mult_HR_target(model, n_traj=100, n_iter=3000, start_temperature=0.5)
#  test_run <- e2e_optimize_act(model, selection="ECO", n_iter=500, start_temperature=0.4 , cooling=0.975, csv.output=TRUE, nyears=40 )
#  #
#  #
#  # Quick Demonstration of optimizing to harvest ratios:
#  test_run <- e2e_optimize_act(model, selection="HR", n_iter=100, start_temperature=1.0, cooling=0.985, csv.output=FALSE, n_traj=5 )
#  str(test_run,max.level=1)                         # View the structure of the returned list
#  str(test_run$new_parameter_data,max.level=1)      # View the structure of the returned list element containing parameter objects
#  test_run$new_parameter_data                       # View the new, final accepted parameter data#
#  #
#  # More realistic configuration would be (WARNING - this will take about 6 hours to run) :
#  model<-e2e_read(model.name="North_Sea",
#                  model.variant="1970-1999",
#                  model.ident="fittingrun",
#                  user.path="C:/Users/username/Documents/Models")
#  fitting_data<-e2e_optimize_act(model, selection="HR", n_iter=3000, start_temperature=1.0, cooling=0.985, csv.output=TRUE, n_traj=100 )

## ----optdiagnostics1, eval = FALSE--------------------------------------------
#  # Load the 1970-1999 version of the North Sea model supplied with the package and generate a quick test data object
#  # with only 10 itereations and running the model for 5 years. More realistic would be at least 500 iterations and
#  # running for 50 years. Even so this example will take a few minutes to run:
#  model<-e2e_read(model.name="North_Sea",
#                  model.variant="1970-1999",
#                  model.ident="test")
#  # This model is already optimized to the observed ecosystem data supplied with the package
#  # Perturb the temperature driving to knock the model away from its maximum likelihood state relative to the target data:
#  model$data$physics.drivers$so_temp <- model$data$physics.drivers$so_temp+3   # add 3 degC to upper layer offshore temperatures
#  model$data$physics.drivers$si_temp <- model$data$physics.drivers$si_temp+3   # add 3 degC to inshore temperatures
#  model$data$physics.drivers$d_temp  <- model$data$physics.drivers$d_temp+3    # add 3 degC to lower layer offshore temperatures
#  test_run  <- e2e_optimize_eco(model, nyears=5, n_iter=10, start_temperature=0.4,csv.output=FALSE,runtime.plot=FALSE)
#  plot_data <- e2e_plot_opt_diagnostics(model,selection="ECO",results=test_run)
#  str(plot_data,max.level=1)  # show the structure of the list object plot_data

## ----optdiagnostics2, fig.width=6, fig.height=8, fig.align='center'-----------
# Or... plot example date supplied with the package showing some data generated during the process of optimizing the North Sea model:
model     <- e2e_read(model.name="North_Sea", model.variant="1970-1999")
plot_data <- e2e_plot_opt_diagnostics(model,selection="ECO",use.example=TRUE)

## ----optdiagnostics3, eval = FALSE--------------------------------------------
#  # Same for harvest ratio optimization...
#  model<-e2e_read(model.name="North_Sea",
#                  model.variant="1970-1999",
#                  model.ident="test")
#  # This model is already optimized to the observed ecosystem data supplied with the package
#  # Perturb the temperature driving to knock the model away from its maximum likelihood state relative to the target data:
#  model$data$physics.drivers$so_temp <- model$data$physics.drivers$so_temp+3   # add 3 degC to upper layer offshore temperatures
#  model$data$physics.drivers$si_temp <- model$data$physics.drivers$si_temp+3   # add 3 degC to inshore temperatures
#  model$data$physics.drivers$d_temp  <- model$data$physics.drivers$d_temp+3    # add 3 degC to lower layer offshore temperatures
#  test_run  <- e2e_optimize_hr(model, nyears=5, n_iter=10, start_temperature=0.4,csv.output=FALSE,runtime.plot=FALSE)
#  plot_data <- e2e_plot_opt_diagnostics(model,selection="HR",results=test_run)
#  str(plot_data,max.level=1)  # show the structure of the list object plot_data

## ----optdiagnostics4, fig.width=6, fig.height=6, fig.align='center'-----------
# Or... plot example date supplied with the package showing some data generated during the process of optimizing the North Sea model:
model     <- e2e_read(model.name="North_Sea", model.variant="2003-2013")
plot_data <- e2e_plot_opt_diagnostics(model,selection="HR",use.example=TRUE)

## ----optdiagnostics5, fig.width=6, fig.height=6, fig.align='center'-----------
# For activity rate optimization relative to ecosystem data:
model<-e2e_read(model.name="North_Sea",
                model.variant="1970-1999",
                model.ident="test")
# This model is already optimized to the observed ecosystem data supplied with the package, but not by optimizing gear activity rates
# The e2e_optimize_eco() function was used in this case.
# Perturb the temperature driving to knock the model away from its maximum likelihood state relative to the target data:
model$data$physics.drivers$so_temp <- model$data$physics.drivers$so_temp+3   # add 3 degC to upper layer offshore temperatures
model$data$physics.drivers$si_temp <- model$data$physics.drivers$si_temp+3   # add 3 degC to inshore temperatures
model$data$physics.drivers$d_temp  <- model$data$physics.drivers$d_temp+3    # add 3 degC to lower layer offshore temperatures
test_run  <- e2e_optimize_act(model, selection="ECO", n_iter=10, start_temperature=0.4, , cooling=0.975, csv.output=FALSE, nyears=5 )
plot_data <- e2e_plot_opt_diagnostics(model,selection="ACT",fitted.to="ECO",results=test_run)
str(plot_data,max.level=1)  # show the structure of the list object plot_data

# There are no example data available in the package for this function

## ----optdiagnostics6, fig.width=6, fig.height=6, fig.align='center'-----------
# For activity rate optimization relative to zonal harvest ratios:
model<-e2e_read(model.name="North_Sea",
                model.variant="1970-1999",
                model.ident="test")
# Activity rates in this model are already optimized to the target harvest ratios supplied with the package but we woud not expect 
# to recover these values in this short demonstration run
test_run  <- e2e_optimize_act(model, selection="HR", n_iter=100, start_temperature=1.0, , cooling=0.985, csv.output=FALSE, n_traj=5 )
plot_data <- e2e_plot_opt_diagnostics(model,selection="ACT",fitted.to="HR",results=test_run)
str(plot_data,max.level=1)  # show the structure of the list object plot_data

## ----optdiagnostics7, fig.width=6, fig.height=6, fig.align='center'-----------
# Or... plot example date supplied with the package showing some data generated during the process of optimizing the North Sea model:
model     <- e2e_read(model.name="North_Sea", model.variant="1970-1999")
plot_data <- e2e_plot_opt_diagnostics(model,selection="ACT",fitted.to="HR",use.example=TRUE)

## ----calculatecode1, eval = TRUE----------------------------------------------
# Load the 2003-2013 version of the North Sea model supplied with the package and calculate scaling parameter values:
model <- e2e_read("North_Sea", "2003-2013")
#
scale_values <- e2e_calculate_hrscale(model)
scale_values
#

## ----sensitivityanalysis1, eval = FALSE---------------------------------------
#  # Load the 2003-2013 version of the North Sea model supplied with the package:
#  model <- e2e_read("North_Sea", "1970-1999")
#  #
#  # Run the sensitivity analysis process:
#  # WARNING - Running a full sensitivity analysis takes days of computer time on a single machine/processor because it involves a huge number of model runs.
#  # The example below is just a (relatively) quick minimalist demonstration and should NOT be taken as the basis for any analysis or conclusions.
#  # Even so, this minimalist demo could take 45 min to run to completion because it involves 1359 model runs.
#  sens_results <- e2e_run_sens(model, nyears=1, n_traj=3, csv.output=FALSE)
#  head(sens_results)
#  #
#  # A more realistic sensitivity analysis would be something like:
#  sens_results <- e2e_run_sens(model, nyears=50, n_traj=16, postprocess=TRUE)
#  # DO NOT launch this configuration unless you are prepared to wait many days for the results
#  #
#  # Example of parallelising the process:
#  # Launch two (or more) runs separately on different processors...
#  # Launch batch 1:
#  model1 <- e2e_read("North_Sea", "1970-1999", model.ident="BATCH1")
#  sens_results1 <- e2e_run_sens(model1, nyears=50, n_traj=10, coldstart=TRUE, postprocess=FALSE)   # On machine 1
#  #
#  model2 <- e2e_read("North_Sea", "1970-1999", model.ident="BATCH2")
#  sens_results2 <- e2e_run_sens(model2, nyears=50, n_traj=10, coldstart=FALSE, postprocess=FALSE)  # On machine 2
#  # Note that these two runs return only raw data since postprocess=FALSE
#  #
#  # Then, afterwards, merge the two raw results files with text-tags BATCH1 and BATCH2, and post process the combined file:
#  model3 <- e2e_read("North_Sea", "1970-1999", model.ident="COMBINED")
#  processed_data <- e2e_merge_sens_mc(model3, selection="SENS", ident.list<-c("BATCH1","BATCH2"), postprocess=TRUE, csv.output=TRUE)
#  #
#  # Plot a diagram of parameter sensitivities from the combined data
#  e2e_plot_sens_mc(model3, selection="SENS", use.example=FALSE)

## ----montecarloanalysis1, eval = FALSE----------------------------------------
#  # Load the 1970-1999 version of the North Sea model supplied with the package:
#     model <- e2e_read("North_Sea", "1970-1999")
#  # Run the Monte Carlo process and generate some data but disable csv output
#  # A quick demonstration of baseline mode with postprocessing and csv output disabled:
#     demo <- e2e_run_mc(model,nyears=2,baseline.mode=TRUE,n_iter=10,csv.output=FALSE)
#        str(demo,max.level=1)                     # View the structure of the returned list
#        str(demo$CI_annual_avmass,max.level=1)    # View the structure of a returned list element containing sub-sets
#        str(demo$CI_annual_fluxes,max.level=1)    # View the structure of a returned list element containing sub-sets
#        head(demo$CI_annual_avmass$whole)         # View the top few rows on the whole domain data on annual averge mass
#  #
#  # --------------------------------------------------------------------------------------
#  #
#  # A more meaningful run would be:
#  # WARNING: This run will take about 48h to complete, much better to split up and spread across multiple processors !
#     mc_results <- e2e_run_mc(model,baseline.mode=TRUE,nyears=50,n_iter=1000)
#  #
#  # --------------------------------------------------------------------------------------
#  #
#  # A quick demonstration run in baseline mode and save the data to csv files:
#     basemodel <- e2e_read("North_Sea", "1970-1999",model.ident="mcbaseline")
#     basedemo <- e2e_run_mc(basemodel,nyears=2,baseline.mode=TRUE,
#                            n_iter=10,csv.output=TRUE)
#  #
#  # --------------------------------------------------------------------------------------
#  #
#  # Then a quick demonstration run in scenario mode using the saved baseline parameter data, and save to csv:
#  # First create an extreme fishing scenario - quadruple some gear activities, run for 10 year
#     scenariomodel<-basemodel
#     scenariomodel$setup$model.ident <- "mcscenario"
#     scenariomodel$data$fleet.model$gear_mult[1] <- 4 # Gear 1 (Pelagic trawls) activity rate rescaled to 4*baseline
#     scenariomodel$data$fleet.model$gear_mult[4] <- 4 # Gear 4 (Beam_Trawl_BT1+BT2) activity rate rescaled to 4*baseline
#     scendemo <- e2e_run_mc(scenariomodel,nyears=10,baseline.mode=FALSE,
#                            use.example.baseparms=FALSE, baseparms.ident="mcbaseline",begin.sample=1,
#                            n_iter=10,csv.output=TRUE)
#  #
#  # Compare the results of the baseline and scenario:
#     basemodel <- e2e_read("North_Sea", "1970-1999", model.ident="mcbaseline")
#     scenariomodel <- e2e_read("North_Sea", "1970-1999",model.ident="mcscenario")
#     e2e_compare_runs_box(selection="ANNUAL", model1=basemodel, ci.data1=TRUE, use.saved1=TRUE,
#  #                        model2=scenariomodel, ci.data2=TRUE, use.saved2=TRUE )
#     dev.new()
#     e2e_compare_runs_box(selection="MONTHLY", model1=basemodel, ci.data1=TRUE, use.saved1=TRUE,
#                          model2=scenariomodel, ci.data2=TRUE, use.saved2=TRUE )
#  #
#  # --------------------------------------------------------------------------------------
#  #
#  # Example of parallelising the process in baseline mode:
#  # Launch batch 1:
#     model1 <- e2e_read("North_Sea", "1970-1999", model.ident="BATCH1")     ### On processor 1
#     results1 <- e2e_run_mc(model1,nyears=2,baseline.mode=TRUE,
#                            n_iter=10,csv.output=TRUE,postprocess=FALSE)
#  # Launch batch 2 (on a different processor):
#     model2 <- e2e_read("North_Sea", "1970-1999", model.ident="BATCH2")     ### On processor 2
#     results2 <- e2e_run_mc(model2,nyears=2,baseline.mode=TRUE,
#                            n_iter=10,csv.output=TRUE,postprocess=FALSE)
#  # Note that these two runs return only raw data since postprocess=FALSE
#  # Note that Batch 2 requests 11 iterations, rather that 10 in Batch 1. The number of iterations do not
#  # have to be the same in each batch. However, the first in each batch has to use the initial parameter set
#  # from the model setup, as this is the parent for all the subsequent samples generated during the run. When we
#  # come to merge the data from separate batches, data from the first iteration are stripped off and discarded for
#  # all but the first batch as we do not want to include duplicate data in the combined files. Hence we choose 11
#  # iterations here in Batch 2 to make the point, and we expect the combined data to include 20 iterations.
#  
#  # Then, afterwards, merge the two raw results files with text-tags BATCH1 and BATCH2, and post process the combined file:
#     model3 <- e2e_read("North_Sea", "1970-1999", model.ident="COMBINED")
#     processed_data <- e2e_merge_sens_mc(model3, selection="MC", ident.list<-c("BATCH1","BATCH2"), postprocess=TRUE, csv.output=TRUE)
#  # or...
#     combined_data <- e2e_merge_sens_mc(model3, selection="MC", ident.list<-c("BATCH1","BATCH2"), postprocess=FALSE, csv.output=TRUE)
#     processed_data<-e2e_process_sens_mc(model3, selection="MC", use.example=FALSE,csv.output=TRUE)
#  #
#  # Plot the parameter likelihood distributions from the combined data
#     e2e_plot_sens_mc(model3, selection="MC")
#  #
#  # --------------------------------------------------------------------------------------
#  #
#  # Example of parallelising the process in scenario mode, using the baseline parameter sets 'COMBINED' from above:
#  # The activity of all fishing gears is reduced to zero to create a no-fishing scenario.
#  # Run each batch for 10 years as a relatively quick demo - a real run would need to run for at least 40 year
#  # Launch batch 1:
#     model1s <- e2e_read("North_Sea", "1970-1999", model.ident="BATCH1_S")     ### On processor 1
#     model1s$data$fleet.model$gear_mult[1:12] <- 0 # Activity rates of all 12 gears rescaled to 0*baseline
#     results1s <- e2e_run_mc(model1s,nyears=10,baseline.mode=FALSE, baseparms.ident="COMBINED", begin.sample=1,
#                            n_iter=10,csv.output=TRUE,postprocess=FALSE)
#  # Launch batch 2 (on a different processor):
#     model2s <- e2e_read("North_Sea", "1970-1999", model.ident="BATCH2_S")     ### On processor 2
#     model2s$data$fleet.model$gear_mult[1:12] <- 0 # Activity rates of all 12 gears rescaled to 0*baseline
#     results2s <- e2e_run_mc(model2s,nyears=10,baseline.mode=FALSE,, baseparms.ident="COMBINED", begin.sample=11,
#                            n_iter=10,csv.output=TRUE,postprocess=FALSE)
#  # Note that Batch 1 samples rows 1:10 of the baseline mode parameter set archive "COMBINED" (begin.sample=1, n_iter=10),
#  # so Batch 2 needs to start sampling at row 11 (begin.sample=11). The baseline archive contains 20 rows, so Batch 2
#  # has the capacity to undertake up to 10 sample iterations (rows 11:20). If we select more than 10 iterations
#  # (e.g. n_iter=15) then the code will automatically restrict to 10. Note that in fact, to be consistent with the
#  # format of output files from the baseline mode, each scenario mode run where 'begin.sample' > 1 will complete
#  # n_iter+1 iterations, by adding an initial run using the parameter values from row 1 of the baseline parameter
#  # set - which is then stripped off during merging.
#  #
#  # Then, merge the two raw results files with text-tags BATCH1_S and BATCH2_S, and post process the combined file:
#     model3s <- e2e_read("North_Sea", "1970-1999", model.ident="COMBINED_S")
#     processed_datas <- e2e_merge_sens_mc(model3s, selection="MC", ident.list<-c("BATCH1_S","BATCH2_S"), postprocess=TRUE, csv.output=TRUE)
#  #
#  # Finally plot comparisons of the baseline and scenario model runs:
#     e2e_compare_runs_box(selection="ANNUAL", model1=model3, ci.data1=TRUE, use.saved1=TRUE,
#                          model2=model3s, ci.data2=TRUE, use.saved2=TRUE )
#     dev.new()
#     e2e_compare_runs_box(selection="MONTHLY", model1=model3, ci.data1=TRUE, use.saved1=TRUE,
#                          model2=model3s, ci.data2=TRUE, use.saved2=TRUE )
#  #

## ----sensmcmerge1, eval = FALSE-----------------------------------------------
#  # Example of parallelising the sensitivity analysis process:
#  # Launch two (or more) runs separately on different processors...
#  # Launch batch 1:
#     model1 <- e2e_read("North_Sea", "1970-1999", model.ident="BATCH1")
#     sens_results <- e2e_run_sens(model1, nyears=50, n_traj=10, coldstart=TRUE, postprocess=FALSE)   # On processor 1
#  # Note that coldstart=TRUE for the first batch only.
#  # Launch batch 2 (on a different processor):
#     model2 <- e2e_read("North_Sea", "1970-1999", model.ident="BATCH2")
#     sens_results <- e2e_run_sens(model2, nyears=50, n_traj=10, coldstart=FALSE, postprocess=FALSE)  # On processor 2
#  # Note that these two runs return only raw data since postprocess=FALSE
#  
#  # Then, afterwards, merge the two raw results files with text-tags BATCH1 and BATCH2, and post process the combined file:
#     model3 <- e2e_read("North_Sea", "1970-1999", model.ident="COMBINED")
#     processed_data <- e2e_merge_sens_mc(model3, selection="SENS", ident.list<-c("BATCH1","BATCH2"), postprocess=TRUE, csv.output=TRUE)
#  # or...
#     combined_data <- e2e_merge_sens_mc(model3, selection="SENS", ident.list<-c("BATCH1","BATCH2"), postprocess=FALSE, csv.output=TRUE)
#     processed_data <- e2e_process_sens_mc(model3, selection="SENS", use.example=FALSE,csv.output=TRUE)
#  #
#  #
#  # Example of parallelising the Monte Carlo process:
#  # Launch batch 1:
#     model1 <- e2e_read("North_Sea", "1970-1999", model.ident="BATCH1")     ### On processor 1
#     results1 <- e2e_run_mc(model1,nyears=2,baseline.mode=TRUE,
#                            n_iter=10,csv.output=TRUE,postprocess=FALSE)
#  # Launch batch 2 (on a different processor):
#     model2 <- e2e_read("North_Sea", "1970-1999", model.ident="BATCH2")     ### On processor 2
#     results2 <- e2e_run_mc(model2,nyears=2,baseline.mode=TRUE,
#                            n_iter=11,csv.output=TRUE,postprocess=FALSE)
#  # Note that these two runs return only raw data since postprocess=FALSE
#  # Note 11 iterations in batch 2 - the first iteration wil be stripped off at merging so the
#  # combined data should include only 20 iterations.
#  
#  # Then, afterwards, merge the two raw results files with text-tags BATCH1 and BATCH2, and post process the combined file:
#     model3 <- e2e_read("North_Sea", "1970-1999", model.ident="COMBINED")
#     processed_data <- e2e_merge_sens_mc(model3, selection="MC", ident.list<-c("BATCH1","BATCH2"), postprocess=TRUE, csv.output=TRUE)
#  # or...
#     combined_data <- e2e_merge_sens_mc(model3, selection="MC", ident.list<-c("BATCH1","BATCH2"), postprocess=FALSE, csv.output=TRUE)
#     processed_data<-e2e_process_sens_mc(model3, selection="MC", use.example=FALSE,csv.output=TRUE)
#  
#  #

## ----sensmcprocess1, eval = FALSE---------------------------------------------
#  # Sensitivity analysis processing:
#  # Load details of the 1970-1999 version of the North Sea model supplied with the package:
#  model <- e2e_read("North_Sea", "1970-1999")
#  # Process the example data for this model variant provided with the package
#     sens_results <- e2e_process_sens_mc(model, selection="SENS", use.example=TRUE)
#     head(sens_results)
#  #'
#  # Monte Carlo analysis processing
#  # The example here assumes that raw results data have been previously
#  # generated by a of the e2e_run_mc() function, with the model.ident value
#  # assigned as "testdata", or that data from parallel runs have been gathered
#  # together in the same folder with model.ident="testdata".
#  # Load the model name and version for the data to be processed :
#  # e.g. ... model <- read_model("North_Sea", "2003-2013", model.ident="testdata")
#     mc_results <- e2e_process_sens_mc(model, selection="MC", csv.output=TRUE)
#     str(mc_results,max.level=1)
#  #

## ----sensmcplot1, fig.width=7, fig.height=7, fig.align='center'---------------
# Load the 1970-1999 version of the North Sea model supplied with the package:
   model <- e2e_read("North_Sea", "1970-1999")
#
# Either run the functions e2e_run_sens() to generate some results, or use the internal example data
#
# Plot the ecology model parameter distributions:
   e2e_plot_sens_mc(model, selection="SENS", use.example=TRUE)  

## ----sensmcplot2, fig.width=7, fig.height=9, fig.align='center'---------------
# Load the 1970-1999 version of the North Sea model supplied with the package:
   model <- e2e_read("North_Sea", "1970-1999")
#
# Either run the functions e2e_run_mc() to generate some results, or use the internal example data
#
# Plot the ecology model parameter distributions:
   e2e_plot_sens_mc(model, selection="MC", use.example=TRUE)  

## ----compareobs1, fig.width=7, fig.height=7, fig.align='center'---------------
# Load the 1970-1999 version of the North Sea model supplied with the package, run, and generate a plot:
# Note that these are the observational data that were used as the target for optimizing the model parameters
model <- e2e_read("North_Sea", "1970-1999")
results <- e2e_run(model, nyears=3)
e2e_compare_obs(selection="ANNUAL", model, results=results)

## ----compareobs2, fig.width=7, fig.height=7, fig.align='center'---------------
# Compare results from the 1970-1999 version of the North Sea model with monthly averaged data:
# Note that these are the observational data that were NOT used for optimizing the model parameters
e2e_compare_obs(model, selection="MONTHLY",results=results)

## ----compareobs3, fig.width=7, fig.height=7, fig.align='center'---------------
# As above, but plotting credible intervals of model data derived from a baseline mode Monte Carlo analysis:
model <- e2e_read("North_Sea", "1970-1999")
e2e_compare_obs(model, selection="ANNUAL",ci.data=TRUE,use.example=TRUE)

## ----compareobs4, fig.width=7, fig.height=7, fig.align='center'---------------
e2e_compare_obs(selection="MONTHLY", model, ci.data=TRUE,use.example=TRUE)

## ----comparerunsbox1, fig.width=7, fig.height=7, fig.align='center'-----------
# Load the 1970-1999 version of the North Sea model supplied with the package and run for 3 years
m1 <- e2e_read("North_Sea", "1970-1999")
r1<-e2e_run(m1,nyears=3)
# Load the 2003-2013 version of the North Sea model supplied with the package and run for 3 years
m2 <- e2e_read("North_Sea", "2003-2013")
r2<-e2e_run(m2,nyears=3)
#Compare 1970-1999 as baseline (from example data with cred.int.), with 2003-2013 as sceanrio (from single model run)
e2e_compare_runs_box(selection="ANNUAL",  model1=m1, ci.data1=TRUE, use.example1=TRUE, model2=m2, ci.data2=FALSE, results2=r2)

## ----comparerunsbox2, fig.width=7, fig.height=7, fig.align='center'-----------
e2e_compare_runs_box(selection="MONTHLY", model1=m1, ci.data1=TRUE, use.example1=TRUE, model2=m2, ci.data2=FALSE, results2=r2)

## ----comparerunsbox3, fig.width=7, fig.height=7, fig.align='center'-----------
#Compare 1970-1999 as baseline (from single model run), with 2003-2013 as sceanrio (from example data with cred.int.)
e2e_compare_runs_box(selection="ANNUAL",  model1=m1, ci.data1=FALSE, results1=r1, model2=m2, ci.data2=TRUE, use.example2=TRUE)

## ----comparerunsbox4, fig.width=7, fig.height=7, fig.align='center'-----------
e2e_compare_runs_box(selection="MONTHLY", model1=m1, ci.data1=FALSE, results1=r1, model2=m2, ci.data2=TRUE, use.example2=TRUE)

## ----comparerunsbox5, fig.width=7, fig.height=7, fig.align='center'-----------
#Compare 1970-1999 as baseline (from example data with cred.int.), with 2003-2013 as sceanrio (from example data with cred.int.)
e2e_compare_runs_box(selection="ANNUAL", model1=m1, ci.data1=TRUE, use.example1=TRUE, model2=m2, ci.data2=TRUE, use.example2=TRUE )

## ----comparerunsbox6, fig.width=7, fig.height=7, fig.align='center'-----------
e2e_compare_runs_box(selection="MONTHLY", model1=m1, ci.data1=TRUE, use.example1=TRUE, model2=m2, ci.data2=TRUE, use.example2=TRUE )

## ----comparerunsbar1, fig.width=7, fig.height=7, fig.align='center'-----------
# Load the 1970-1999 version of the North Sea model supplied with the package :
base_model   <- e2e_read("North_Sea", "1970-1999",model.ident="baseline")
base_results <- e2e_run(base_model,nyears=3)
#
# Create a scenario run from 1970-1999 baseline:
scen1_model   <- base_model      # Copies the baseline configuration into a new model object
scen1_model$setup$model.ident <- "scenario1"
scen1_model$data$fleet.model$gear_mult[4] <- 0.5 # Gear 4 (Beam_Trawl_BT1+BT2) activity rate rescaled to 0.5*baseline
scen1_results <- e2e_run(scen1_model,nyears=30)
#
#Compare the annual average mass scenario data with the baseline
mdiff_results1 <- e2e_compare_runs_bar(selection="AAM",
                                       model1=NA, use.saved1=FALSE, results1=base_results,
				       model2=NA, use.saved2=FALSE, results2=scen1_results,
   	                               log.pc="PC", zone="W",
                 		       bpmin=(-40),bpmax=(+40),
       		                       maintitle="Beam Trawl activity reduced by half")

## ----comparerunsbar2, eval = TRUE---------------------------------------------
str(mdiff_results1,max.level=1)   # View the structure of the returned list object

## ----comparerunsbar3, fig.width=7, fig.height=7, fig.align='center'-----------
#Create a second sceanario run
scen2_model   <- base_model     # Copies the baseline configuration into a new model object
scen2_model$setup$model.ident <- "scenario2"
scen2_model$data$fleet.model$gear_mult[1] <- 0.5 # Gear 1 (Pelagic_Trawl+Seine) activity rate rescaled to 0.5*baseline
scen2_results <- e2e_run(scen2_model,nyears=30)
#
#Compare the annual catches in the new scenario with the baseline previously saved in a csv files
mdiff_results2 <- e2e_compare_runs_bar(selection="CATCH",
			              model1=base_model, use.saved1=TRUE, results1=NA,
                     	              model2=NA,use.saved2=FALSE, results2=scen2_results,
                     	              log.pc="LG", zone="W",
                     	              bpmin=(-0.4),bpmax=(+0.6),
                     	              maintitle="Pelagic Trawl/Seine activity reduced by half")

## ----runycurve, eval = FALSE--------------------------------------------------
#  # Load the 1970-1999 version of the North Sea model supplied with the package :
#  model <- e2e_read("North_Sea", "1970-1999",model.ident="70-99base")
#  #
#  # In this illustrative example the StrathE2E() model is run for only 5 years to enable quick return of results. In a real simulation nyear would be at least 50.
#  # This example illustrates that the vector of planktivorous fish harvest ratio multiplers does not have to be evenly spaced.
#  hr=c(0,0.5,0.75,1.0,1.25,2.0,3.0)
#  pf_yield_data <- e2e_run_ycurve(model,selection="PLANKTIV", nyears=5,HRvector=hr,HRfixed=1,csv.output=TRUE)
#  #
#  # View the column names of the results dataframe:
#  names(pf_yield_data)
#  #

## ----plotycurve1, fig.width=7, fig.height=7, fig.align='center'---------------
# Load the 1970-1999 version of the North Sea model supplied with the package and generate a yield data object:
# WARNING - this example will take about several minutes to run
model <- e2e_read("North_Sea", "1970-1999")
pfhr=c(0,0.5,0.75,1.0,1.25,1.5,2.0,2.5,3.0) 
pf_yield_data <- e2e_run_ycurve(model,selection="PLANKTIV",nyears=50,HRvector=pfhr,HRfixed=1,csv.output=FALSE)
data <- e2e_plot_ycurve(model,selection="PLANKTIV", results=pf_yield_data,
                        title="Planktivorous yield with baseline demersal fishing")

## ----plotycurve2, fig.width=7, fig.height=7, fig.align='center'---------------
# Using examaple data generated with selection="PLANKTIV" ...
# Plot example data for one of the North Sea model versions internal to the package
model <- e2e_read("North_Sea", "1970-1999")
pf_yield_data<-e2e_plot_ycurve(model, selection="PLANKTIV", use.example=TRUE)

## ----plotycurve3, fig.width=7, fig.height=7, fig.align='center'---------------
# Users can then plot other biomass, landings and discards data in the results object by, for example:
par(mfrow=c(2,1))
par(mar=c(3.2,5,2,0.8))
ym<-1.1*max(pf_yield_data$Cetaceanbiom)
plot(pf_yield_data$PlankFishHRmult,pf_yield_data$Cetaceanbiom,ylim=c(0,ym),type="l",lwd=3,yaxt="n",xaxt="n",ann=FALSE)
   abline(v=1,lty="dashed")
   axis(side=1,las=1,cex.axis=0.9)
   axis(side=2,las=1,cex.axis=0.9)
   mtext("Planktiv. fish harvest ratio multiplier",cex=1,side=1,line=2)
   mtext("Cetacean biomass",cex=1,side=2,line=3.5)
   mtext(bquote("mMN.m"^-2),cex=0.7,side=3,line=-0.05,adj=-0.18)
ym<-1.1*max(pf_yield_data$Cetaceandisc)
plot(pf_yield_data$PlankFishHRmult,pf_yield_data$Cetaceandisc,ylim=c(0,ym),type="l",lwd=3,yaxt="n",xaxt="n",ann=FALSE)
   abline(v=1,lty="dashed")
   axis(side=1,las=1,cex.axis=0.9)
   axis(side=2,las=1,cex.axis=0.9)
   mtext("Planktiv. fish harvest ratio multiplier",cex=1,side=1,line=2)
   mtext("Cetacean by-catch",cex=1,side=2,line=3.5)
   mtext(bquote("mMN.m"^-2 ~ ".y"^-1),cex=0.7,side=3,line=-0.05,adj=-0.18)

## ----plotycurve4, fig.width=7, fig.height=7, fig.align='center'---------------
# Using example data generated with selection="DEMERSAL"...
model <- e2e_read("North_Sea", "1970-1999")
df_yield_data<-e2e_plot_ycurve(model, selection="DEMERSAL", use.example=TRUE)

## ----plotycurve5, fig.width=7, fig.height=7, fig.align='center'---------------
# Users can then plot other biomass, landings and discards data in the results object by, for example:
par(mfrow=c(2,1))
par(mar=c(3.2,5,2,0.8))
ym<-1.1*max(df_yield_data$Pinnipedbiom)
plot(df_yield_data$DemFishHRmult,df_yield_data$Pinnipedbiom,ylim=c(0,ym),type="l",lwd=3,yaxt="n",xaxt="n",ann=FALSE)
    abline(v=1,lty="dashed")
    axis(side=1,las=1,cex.axis=0.9)
    axis(side=2,las=1,cex.axis=0.9)
    mtext("Demersal fish harvest ratio multiplier",cex=1,side=1,line=2)
    mtext("Pinniped biomass",cex=1,side=2,line=3.7)
    mtext(bquote("mMN.m"^-2),cex=0.7,side=3,line=-0.05,adj=-0.18)
ym<-1.1*max(df_yield_data$Pinnipeddisc)
plot(df_yield_data$DemFishHRmult,df_yield_data$Pinnipeddisc,ylim=c(0,ym),type="l",lwd=3,yaxt="n",xaxt="n",ann=FALSE)
    abline(v=1,lty="dashed")
    axis(side=1,las=1,cex.axis=0.9)
    axis(side=2,las=1,cex.axis=0.9)
    mtext("Demersal fish harvest ratio multiplier",cex=1,side=1,line=2)
    mtext("Pinniped by-catch",cex=1,side=2,line=3.7)
    mtext(bquote("mMN.m"^-2 ~ ".y"^-1),cex=0.7,side=3,line=-0.05,adj=-0.18)

## ----plotenvdrivers, fig.width=7, fig.height=7, fig.align='center'------------
model<-e2e_read(model.name="North_Sea", model.variant="2003-2013")
e2e_plot_edrivers(model)

## ----plotfdrivers1, fig.width=7, fig.height=6, fig.align='center'-------------
# Load the 1970-1999 version of the North Sea model supplied with the package, run, and generate a plot:
model <- e2e_read("North_Sea", "2003-2013")
plotted_data<-e2e_plot_fdrivers(model, selection="ACTIVITY")

## ----plotfdrivers2, fig.width=7, fig.height=6, fig.align='center'-------------
# Load the 1970-1999 version of the North Sea model supplied with the package, run, and generate a plot:
model <- e2e_read("North_Sea", "2003-2013")
plotted_data<-e2e_plot_fdrivers(model, selection="ABRASION")

## ----plotfdrivers3, fig.width=7, fig.height=6, fig.align='center'-------------
# Load the 1970-1999 version of the North Sea model supplied with the package, run, and generate a plot:
model <- e2e_read("North_Sea", "2003-2013")
plotted_data<-e2e_plot_fdrivers(model, selection="HARVESTR")

## ----plotfdrivers4 , fig.width=7, fig.height=6, fig.align='center'------------
# Load the 1970-1999 version of the North Sea model supplied with the package, run, and generate a plot:
model <- e2e_read("North_Sea", "2003-2013")
plotted_data<-e2e_plot_fdrivers(model, selection="DISCARDS")

## ----plotfdrivers5 , fig.width=7, fig.height=6, fig.align='center'------------
# Load the 1970-1999 version of the North Sea model supplied with the package, run, and generate a plot:
model <- e2e_read("North_Sea", "2003-2013")
plotted_data<-e2e_plot_fdrivers(model, selection="OFFAL")

## ----ploteco1 , fig.width=7, fig.height=6, fig.align='center'-----------------
# Load the 1970-1999 version of the North Sea model supplied with the package, run, and generate a plot:
model <- e2e_read("North_Sea", "1970-1999")
results <- e2e_run(model, nyears=3)
e2e_plot_eco(model, selection="NUT_PHYT",results=results)

## ----ploteco2 , fig.width=7, fig.height=7, fig.align='center'-----------------
# Load the 1970-1999 version of the North Sea model supplied with the package and plot example credible interval data:
model <- e2e_read("North_Sea", "1970-1999")
e2e_plot_eco(model, selection="NUT_PHYT", ci.data=TRUE, use.example=TRUE)

## ----ploteco0 , eval=FALSE----------------------------------------------------
#  pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=7,height=4.667)      # or jpeg("plot.jpg"), png("plot.png")
#       e2e_plot_eco(model, "NUT_PHYT", ci.data=TRUE, use.example=TRUE)
#  dev.off()

## ----plotmig1 , fig.width=7, fig.height=7, fig.align='center'-----------------
# Load the 1970-1999 version of the North Sea model supplied with the package, run, and generate a plot:
model <- e2e_read("North_Sea", "1970-1999")
results <- e2e_run(model, nyears=3)
e2e_plot_migration(model, results=results)

## ----plotmig2 , fig.width=7, fig.height=7, fig.align='center'-----------------
# For the same model as supplied with the package, plot the example data with credible intervals:
model <- e2e_read("North_Sea", "1970-1999")
e2e_plot_migration(model, ci.data=TRUE, use.example=TRUE)

## ----plotcatch1 , fig.width=7, fig.height=7, fig.align='center'---------------
# Load the 1970-1999 version of the North Sea model supplied with the package, run, and generate a plot:
model <- e2e_read("North_Sea", "1970-1999")
results <- e2e_run(model, nyears=3)
e2e_plot_catch(model, results, selection="BY_GEAR")

## ----plotcatch2 , fig.width=7, fig.height=7, fig.align='center'---------------
# Load the 1970-1999 version of the North Sea model supplied with the package, run, and generate a plot:
e2e_plot_catch(model, results, selection="BY_GUILD")

## ----plottrophic1 , fig.width=7, fig.height=7, fig.align='center'-------------
# Load the 1970-1999 version of the North Sea model supplied with the package, run, and generate a plot:
model <- e2e_read("North_Sea", "1970-1999")
results <- e2e_run(model, nyears=3)
e2e_plot_trophic(model, results=results)

## ----plottrophic2 , fig.width=7, fig.height=7, fig.align='center'-------------
# For the same model as supplied with the package, plot the example data with credible intervals:
e2e_plot_trophic(model, ci.data=TRUE, use.example=TRUE)

## ----plotbiomass1 , fig.width=7, fig.height=7, fig.align='center'-------------
# Load the 1970-1999 version of the North Sea model supplied with the package, run, and generate a plot:
model <- e2e_read("North_Sea", "1970-1999")
results <- e2e_run(model, nyears=3)
e2e_plot_biomass(model, results=results)

## ----plotbiomass2 , fig.width=7, fig.height=7, fig.align='center'-------------
# For the same model as supplied with the package, plot the example data with credible intervals:
e2e_plot_biomass(model, ci.data=TRUE, use.example=TRUE)

